<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Barcode Print Sheet</title>
    <style>
        @page {
            size: A4;
            margin: 0;
        }
        
        body {
            margin: 0;
            padding: 10mm;
            width: 210mm;
            box-sizing: border-box;
            -webkit-print-color-adjust: exact;
            print-color-adjust: exact;
        }

        .page {
            width: 190mm;
            min-height: 277mm;
            page-break-after: always;
            padding-bottom: 10mm;
        }

        .page:last-child {
            page-break-after: auto;
        }

        .barcode-row {
            display: flex;
            justify-content: flex-start;
            margin-bottom: 5mm;
            gap: 2mm;
        }

        .barcode-item {
            width: 37mm;
            height: 13mm;
            margin: 0;
            padding: 0;
            border: 0.1mm solid #eee;
            background: white;
        }

        .barcode-item img {
            width: 100%;
            height: 100%;
            display: block;
            object-fit: fill;
            image-rendering: -webkit-optimize-contrast;
            image-rendering: crisp-edges;
            -ms-interpolation-mode: nearest-neighbor;
            transform: translateZ(0);
            backface-visibility: hidden;
            -webkit-font-smoothing: subpixel-antialiased;
        }

        @media print {
            body {
                padding: 0;
                margin: 10mm;
            }
            .barcode-item {
                border: none;
            }
            img {
                image-resolution: 300dpi;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }
        }
    </style>
</head>
<body>
    <?php
    // Increase PHP memory limit for large image processing
    ini_set('memory_limit', '256M');
    
    $barcodes = glob("*.gif");
    
    // Sort barcodes based on last 4 numbers in filename
    usort($barcodes, function($a, $b) {
        preg_match('/(\d{4})[^\d]*$/', $a, $matchesA);
        preg_match('/(\d{4})[^\d]*$/', $b, $matchesB);
        
        $numA = isset($matchesA[1]) ? intval($matchesA[1]) : 0;
        $numB = isset($matchesB[1]) ? intval($matchesB[1]) : 0;
        
        return $numA - $numB;
    });

    $total = count($barcodes);
    $barcodesPerPage = 80;
    $totalPages = ceil($total / $barcodesPerPage);

    // Generate pages
    for($pageNum = 0; $pageNum < $totalPages; $pageNum++) {
        echo '<div class="page">';
        
        // Calculate start index for current page
        $startIndex = $pageNum * $barcodesPerPage;
        $endIndex = min($startIndex + $barcodesPerPage, $total);
        
        // Create rows with exactly 5 barcodes each for current page
        for ($i = $startIndex; $i < $endIndex; $i += 5) {
            echo '<div class="barcode-row">';
            
            // Add 5 barcodes to this row or remaining barcodes if less than 5
            for ($j = 0; $j < 5 && ($i + $j) < $endIndex; $j++) {
                $barcode = $barcodes[$i + $j];
                echo '<div class="barcode-item">';
                echo '<img src="' . htmlspecialchars($barcode) . '" alt="Barcode" ' .
                     'loading="eager" decoding="sync" ' .
                     'style="image-rendering: -webkit-optimize-contrast; transform: translateZ(0);">';
                echo '</div>';
            }
            
            echo '</div>';
        }
        
        echo '</div>';
    }
    ?>
    <script>
        window.onload = function() {
            // Force browser to load images before printing
            Promise.all(Array.from(document.images).map(img => {
                if (img.complete) return Promise.resolve();
                return new Promise(resolve => img.onload = resolve);
            })).then(() => {
                window.print();
            });
        };
    </script>
</body>
</html>